# OTP Builder Toolkit
**Self-Made One-Time Pads from Everyday Sources – Practical Generation & Distribution**  
*Guerrilla Encryption: Making the Cipher the Target*  

---

## Table of Contents
1. [Overview](#overview)  
2. [Folder Structure](#folder-structure)  
3. [Prerequisites / Dependencies](#prerequisites--dependencies)  
4. [Installation & Setup](#installation--setup)  
5. [Quick Start – End-to-End Example](#quick-start--end-to-end-example)  
6. [Detailed Script Reference](#detailed-script-reference)  
7. [Generating the OTP Booklet (PDF)](#generating-the-otp-booklet-pdf)  
8. [Creating a QR Code (optional)](#creating-a-qr-code-optional)  
9. [Testing Randomness](#testing-randomness)  
10. [Bait-And-Switch Playbook (quick recap)]
11. [License](#license)  

---

## Overview  

This toolkit contains a small collection of Bash and Python scripts that let you:

* Turn physical entropy (dice rolls, coin flips, playing-card draws) into a **binary one-time-pad (OTP)**.  
* Verify that the generated keystream has proper entropy.  
* Encrypt a plaintext message with the OTP (XOR) and wrap the result in a **fake PGP block**.  
* Decrypt the ciphertext using the matching OTP line.  
* Produce a printable **3×5in pocket-size OTP booklet** from a CSV of symbols.  
* Optionally encode any text file as a QR code for easy transfer to a phone.  

All scripts are **self-contained**, require only standard Linux utilities plus a tiny Python dependency (`reportlab`), and work on any recent Pop!\_OS, Ubuntu, or other Debian-based distro.

---

## Folder Structure  

otp-builder/
│
├─ README.md                # ← you are here
├─ LICENSE                  # MIT licence
│
├─ scripts/                 # Executable helpers
│   ├─ build_pad.sh         # OTP generation + XOR encryption
│   ├─ decode_pad.sh        # OTP decryption (reverse of build_pad.sh)
│   ├─ b64_encode.sh        # Base-64 encode a binary file
│   ├─ wrap_pgp.sh          # Add fake PGP delimiters
│   ├─ entropy_check.sh     # Run ent + rngtest on a keystream
│   ├─ pdf_bookmaker.py     # Create the 3×5 in OTP booklet PDF
│   ├─ qr_encode.sh         # (optional) QR-code generator
│   └─ utils/               # Mapping tables used by the scripts
│       ├─ card_map.txt     # A♣ → 0 … K♠ → 51 (6-bit)
│       ├─ die_map.txt      # 1-6 → 0-5 (3-bit)
│       └─ coin_map.txt     # H → 0, T → 1 (1-bit)
│
├─ examples/                # Sample inputs for quick testing
├─ sample_symbols.csv       # 10 × 30 grid of raw symbols
├─ dummy_line.txt       # One example OTP line (raw symbols)
└─ dummy_message.txt    # Short plaintext for test encryptions


##Prerequisites / Dependencies
Package	                  Install command (Pop!_OS / Ubuntu)
ent (entropy estimator)	      sudo apt install ent
rngtest (from rng-tools)	sudo apt install rng-tools
qrencode (QR code generator)	sudo apt install qrencode
steghide (LSB steganography)	sudo apt install steghide
python3 (>= 3.6)	            pre-installed
pip3 (Python package manager)	pre-installed
reportlab (PDF gen library)	pip3 install --user reportlab
Standard GNU utilities (awk, sed, base64, xxd, cut, head, dd)	pre-installed

All other tools used (bash, cat, rm, etc.) are part of any typical Linux shell.
Installation & Setup

1️⃣  Download / extract the toolkit
wget https://example.com/otp-builder.zip   # replace with your actual URL
unzip otp-builder.zip
cd otp-builder

2️⃣  Install the Python dependency (reportlab)
pip3 install --user reportlab

3️⃣  Make the scripts executable (once per checkout)
chmod +x scripts/*.sh
chmod +x scripts/pdf_bookmaker.py

###Normalization Support (new)

The toolkit now automatically normalizes card tokens so you can write them in any of the following ways without breaking the conversion:

    -Upper- or lower-case (ac, Ac, AC).
    -Ten represented as T or 10 (TC, TH, 10C).
    -Full-word suits (clubs, diamonds, hearts, spades).

The scripts (build_pad.sh and decode_pad.sh) contain a small helper function norm_card() that:

    Upper-cases the token.
    Replaces a lone T followed by a suit letter with 10 + suit.
    Translates full-word suit names to their single-letter abbreviations.

Because the normalization happens before the lookup in card_map.txt, you can keep your printed OTP booklet in the style you prefer (e.g., 10C, AD, QS, 9D) and still type ac, tenc, or clubs on the command line without editing the mapping file.

If you prefer a completely custom notation, simply edit scripts/utils/card_map.txt to match the tokens you will write; the normaliser will still handle case and suit-word variations.

Quick Start – End-to-End Example

Below is a minimal workflow that goes from a raw OTP line to a ready-to-send SMS payload.

1️⃣  Export a fresh OTP line (copy it from your printed booklet)
echo "A♣ 7H 3D 9C 5S AD 2C QH 8D JC" > line.txt

2️⃣  Write the plaintext you want to send
echo "Meet me at the cafe on Main Street at 3 pm." > plain.txt

3️⃣  Encrypt (OTP XOR) → binary ciphertext
./scripts/build_pad.sh line.txt plain.txt ct.bin

4️⃣  Verify the keystream quality (optional but recommended)
./scripts/entropy_check.sh ct.bin   
-look for entropy ≈ 8bits/byte, p>0.01

5️⃣  Base-64 encode the ciphertext
./scripts/b64_encode.sh ct.bin ct.b64

6️⃣  Wrap in a fake PGP block (this is what the attacker will see)
./scripts/wrap_pgp.sh ct.b64 real_msg.txt

7️⃣  OPTIONAL: create a QR code for phone-only transfer
./scripts/qr_encode.sh real_msg.txt msg_qr.png
-Scan `msg_qr.png` on your phone; the QR scanner will copy the whole block to the clipboard.

8️⃣  Transfer `real_msg.txt` to your phone (Syncthing, QR, USB, etc.) and paste it into an SMS.
#    Hit Send – the message now looks like a normal PGP encrypted block.

Recipient side (assuming they have the matching line in their own booklet):

-Export the same line from their booklet
echo "A♣ 7H 3D 9C 5S AD 2C QH 8D JC" > recv_line.txt

-Decrypt the received PGP block
./scripts/decode_pad.sh recv_line.txt real_msg.txt recovered.txt

# View the plaintext
cat recovered.txt
# → Meet me at the cafe on Main Street at 3 pm.

That’s the entire “real-message” flow. The next section shows how to create the decoy bait that forces an adversary to waste resources.

Detailed Script Reference Script	

Description	Key Arguments	
Example

build_pad.sh	Convert raw symbols → binary keystream → XOR with plaintext. Handles normalization (case, T→10, full-word suits).	<line_file> <plaintext_file> <output_cipher_bin>	./scripts/build_pad.sh line.txt plain.txt ct.bin
decode_pad.sh	Reverse of build_pad.sh (strip PGP delimiters, Base-64 decode, XOR). Includes the same normalisation step.	<line_file> <pgp_wrapped_message> <output_plaintext>	./scripts/decode_pad.sh line.txt real_msg.txt out.txt
b64_encode.sh	Base-64 encode a binary file (single-line output).	<binary_input> <b64_output>	./scripts/b64_encode.sh ct.bin ct.b64
wrap_pgp.sh	Add -----BEGIN PGP MESSAGE----- / -----END PGP MESSAGE----- around a Base-64 string.	<b64_input> <pgp_output>	./scripts/wrap_pgp.sh ct.b64 real_msg.txt
entropy_check.sh	Run ent and rngtest on a binary file; prints a concise pass/fail summary.	<binary_file>	./scripts/entropy_check.sh ct.bin
pdf_bookmaker.py	Generate a 3×5 in OTP booklet PDF from a CSV of symbols.	<csv_input> <output_pdf>	python3 scripts/pdf_bookmaker.py examples/sample_symbols.csv otp_book.pdf
qr_encode.sh (optional)	Encode any text file as a QR-code PNG (useful for phone-only transfer).	<text_input> <png_output>	./scripts/qr_encode.sh real_msg.txt msg_qr.png

All scripts read the mapping tables located in scripts/utils/ automatically; you do not need to edit them unless you want a custom mapping.
Generating the OTP Booklet (PDF)

    Prepare a CSV where each line corresponds to a row in the booklet, and each cell is a raw symbol (cards, dice, or coins).
    You can start from the provided example:

    cp examples/sample_symbols.csv my_symbols.csv

Run the PDF generator:

python3 scripts/pdf_bookmaker.py my_symbols.csv otp_book.pdf

    Print otp_book.pdf on 250gsm matte cardstock (or any heavy paper that your printer can handle).

    Cut along the tiny crop-marks (the script adds a cross-hair at each corner). Scissors, a paper cutter, or a rotary trimmer work fine.

    Bind the pages (staple along the short edge, or use a small ring binder).

Each printed row is now a single-use OTP line. When you finish a message, tear that row out and destroy it (shred or burn).

##Creating a QR Code (Optional)

If you cannot type the PGP block on a phone, generate a QR code:

./scripts/qr_encode.sh real_msg.txt msg_qr.png

Open the image on any device, scan it with a QR scanner (most smartphones have this built-in), and the scanner will copy the entire block to the clipboard. Paste it into your SMS app and send.

##Testing Randomness

Before you trust a newly generated OTP line, run:

./scripts/entropy_check.sh pad.bin

Typical successful output:

[ENT]   Entropy = 7.999 bits per byte   (ideal = 8.0)
[ENT]   Optimum compression = 0.0%
[ENT]   Chi-square distribution = 0.42 (p = 0.42)   <-- p > 0.01 = OK
[RNGTEST]  p-value = 0.67   <-- p > 0.01 = OK

If either p-value is ≤0.01, collect more physical entropy (additional dice rolls, coin flips, or card draws) and rebuild the line.

##Bait-And-Switch Playbook (Quick Recap)

Phase	Goal	Core Commands

Preparation (decoy)	Create a single-use OTP, hide it in a PNG/PDF, package in a password-protected ZIP, publish on a no-login paste service.	build_pad.sh → steghide → zip -e → curl -F

Deployment (real message)	Encrypt the actual message with a different OTP line, wrap in fake PGP, send via SMS/email.	build_pad.sh → b64_encode.sh → wrap_pgp.sh → Syncthing/QR → SMS
Cleanup	Destroy used OTP lines, delete local bait files, rotate the bait weekly.	Manual tear-out + rm + repeat preparation.

Publishing the bait forces any data-broker or surveillance system to run password-crackers, stego-analysis tools, and OpenPGP parsers—none of which succeed because the hidden OTP is already burned. Meanwhile the genuine OTP-encrypted message reaches the intended recipient with perfect secrecy.

##License

This toolkit is released under the MIT License. See the LICENSE file for the full text. Feel free to fork, modify, and redistribute the code, provided you retain the original copyright notice.

Enjoy building your own guerrilla-encryption system!
If you run into any issues, check the examples/ directory for ready-made inputs, or open an issue on the repository page.

Note: The scripts now include automatic normalization of card tokens (case-insensitivity, T → 10, full-word suits). This makes it safe to write cards in any of the common notations (10C, AD, QS, 9D, clubs, etc.) without having to edit the mapping file each time. The normalization logic lives at the top of build_pad.sh and decode_pad.sh.
